# -*- coding: utf-8 -*-

"""Defines the StagedMoviesMenu class."""

import xbmcgui
from resources import ADDON_NAME
from resources.lib.misc import get_string, notification


class StagedMoviesMenu:
    """Provide windows for displaying staged movies, and tools for managing the items."""

    # TODO: don't commit sql changes for "... all" until end
    # TODO: decorator for "...all" commands
    # TODO: load staged movies on init, use as instance variable, refresh as needed

    def __init__(self, database, progressdialog):
        """__init__ StagedMoviesMenu."""
        self.database = database
        self.progressdialog = progressdialog

    def add_all(self, items):
        """Add all staged movies to library."""
        STR_ADDING_ALL_MOVIES = get_string(32042)
        STR_ALL_MOVIES_ADDED = get_string(32043)
        self.progressdialog.create_progressdialog(msg=STR_ADDING_ALL_MOVIES)
        for index, item in enumerate(items):
            self.progressdialog.update_progressdialog(index / len(items), item.title())
            item.add_to_library()
        self.progressdialog.close_progressdialog()
        notification(STR_ALL_MOVIES_ADDED)

    @staticmethod
    def rename_dialog(item):
        """Prompt input for new name, and rename if non-empty string."""
        # TODO: move to utils or parent class so it's not duplicated
        input_ret = xbmcgui.Dialog().input("Title", defaultt=item.title())
        if input_ret:
            item.rename(input_ret)

    def options(self, item):
        """Provide options for a single staged movie in a dialog window."""
        STR_ADD = get_string(32048)
        STR_REMOVE = get_string(32017)
        STR_REMOVE_AND_BLOCK = get_string(32049)
        STR_RENAME = get_string(32050)
        STR_STAGED_MOVIE_OPTIONS = get_string(32053)
        STR_BACK = get_string(32011)
        lines = [
            STR_ADD,
            STR_REMOVE,
            STR_REMOVE_AND_BLOCK,
            # STR_RENAME,
            STR_BACK,
        ]
        ret = xbmcgui.Dialog().select(
            f"{ADDON_NAME} - {STR_STAGED_MOVIE_OPTIONS} - {item.title}", lines
        )
        if ret >= 0:
            if lines[ret] == STR_ADD:
                item.add_to_library()
            elif lines[ret] == STR_REMOVE:
                item.delete()
            elif lines[ret] == STR_REMOVE_AND_BLOCK:
                item.remove_and_block()
            elif lines[ret] == STR_RENAME:
                self.rename_dialog(item)
                self.options(item)
                return
            elif lines[ret] == STR_BACK:
                return
            self.show_all()
        self.show_all()

    def remove_all(self):
        """Remove all staged movies."""
        STR_REMOVING_ALL_MOVIES = get_string(32013)
        STR_ALL_MOVIES_REMOVED = get_string(32014)
        self.progressdialog.create_progressdialog(msg=STR_REMOVING_ALL_MOVIES)
        self.database.delete_item_from_table_with_status_or_showtitle(
            _type="movie", status="staged"
        )
        self.progressdialog.close_progressdialog()
        notification(STR_ALL_MOVIES_REMOVED)

    def show_all(self):
        """
        Display all staged movies, which are selectable and lead to options.

        Also provides additional options at bottom of menu.
        """
        STR_NO_STAGED_MOVIES = get_string(32037)
        STR_ADD_ALL_MOVIES = get_string(32038)
        STR_REMOVE_ALL_MOVIES = get_string(32009)
        STR_BACK = get_string(32011)
        STR_STAGED_MOVIES = get_string(32004)
        staged_movies = list(
            self.database.get_content_items(status="staged", _type="movie")
        )
        if not staged_movies:
            xbmcgui.Dialog().ok(ADDON_NAME, STR_NO_STAGED_MOVIES)
            return
        lines = [str(x) for x in staged_movies]
        lines += [STR_ADD_ALL_MOVIES, STR_REMOVE_ALL_MOVIES, STR_BACK]
        ret = xbmcgui.Dialog().select(f"{ADDON_NAME} - {STR_STAGED_MOVIES}", lines)
        if ret >= 0:
            if ret < len(staged_movies):  # staged item
                for i, item in enumerate(staged_movies):
                    if ret == i:
                        self.options(item)
                        break
            elif lines[ret] == STR_ADD_ALL_MOVIES:
                self.add_all(staged_movies)
            elif lines[ret] == STR_REMOVE_ALL_MOVIES:
                self.remove_all()
            elif lines[ret] == STR_BACK:
                return
